<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\Controller;
use App\Models\Branch;
use App\Models\User;
use App\Models\UserActivityLog;
use App\Notifications\WelcomeNotification;
use App\Rules\Mobile;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class AuthController extends Controller
{

    public function index()
    {
    }

    /**
     * Login
     *
     * @param Request $request
     * @return void
     */
    public function login(Request $request)
    {

        $data['error'] = '';

        if (user_id()) {
            return redirect()->route('admin.dashboard');
        }

        session()->regenerateToken();

        if ($request->getMethod() == 'POST') {

            $rules = [];

            if ( is_numeric($request->input('username') ) ){
                $rules['username'] = ['required',new Mobile];
            }else{
                $rules['username'] = ['required','email'];
            }
            
            $rules['password'] = ['required'];

            $messages = [
                'username.*' => 'Username must be a valid E-mail or mobile number.',
            ];

            $validated = $request->validate($rules,$messages);

            $user_info = User::where('email', $validated['username'])->orWhere('mobile', $validated['username'])->first();

            // verify password
            if ($user_info) {

                if ($user_info->login_attempts >= 5) {

                    $user_info->update(['is_blocked' => 1]);
                    session()->flash('error', 'Your account has been blocked due to too many login attempts');

                } else {

                    $db_pass = $user_info['password'];

                    if (Hash::check($request->input('password'), $db_pass)) {

                        $hashed = md5(md5(session()->token()));


                        session()->put('user_id_' . $hashed, $user_info->id);
                        session()->put('user_name_' . $hashed, $user_info->first_name . ' ' . $user_info->last_name);

                        // set allowed branches
                        $allowed_branches = [];
                        $allowed_regions = [];

                        if ($user_info->is_admin) {

                            $branches = Branch::get();

                            foreach ($branches as $branch) {
                                $allowed_branches[] = $branch->id;
                                $allowed_regions[] = $branch->region_id;
                            }
                        } else {

                            $branches = $user_info->branches;

                            foreach ($branches as $branch) {
                                $allowed_regions[] = $branch->region_id;
                                $allowed_branches[] = $branch->id;
                            }
                        }

                        $user_info->update(['login_attempts' => 0 ]);

                        UserActivityLog::create([
                            'user_id' => user_id(),
                            'route' => $request->route()->getName(),
                            'request_method' => $request->method(),
                        ]);

                        // warning for weak passwords
                        $pass = $request->input('password');
                        
                        $weak_pass = ['123456789','12345678','1234567890','987654321','9876543210','87654321','876543210'];

                        if ( in_array($pass,$weak_pass) ){
                            
                            $msg = 'كلمة المرور ضعيفة و قد تعرض النظام إلى تسريب بيانات. يرجى تغيير كلمة المرور في الحال من خلال  %s';
                        
                            session()->put('danger',sprintf($msg,'<a href="' . route('admin.profile') .'">الضغط هنا</a>'));
                        
                        }else{
                            session()->pull('danger');
                        }
                        

                        return redirect()->route('admin.dashboard');
                        
                    } else {

                        $current_attempt = $user_info->login_attempts;

                        $attempt = $current_attempt + 1;

                        $user_info->update(['login_attempts' => $attempt]);

                        session()->flash('error', 'Invalid login');
                    }
                }
            }else{

                session()->flash('error', 'Invalid login');

            }

        }

        $data['title'] = __('admin/auth.login.title');

        return view('admin.auth.login', $data);
    }
    //----------------------------------------------------------
    /**
     * Logout
     *
     * @return void
     */
    public function logout(Request $request)
    {

        if (user_id()) {

            $hashed = md5(md5(session()->token()));

            session()->pull('user_id_' . $hashed);

            session()->pull('user_name_' . $hashed);

            return redirect()->route('admin.auth.login');
        } else {

            return redirect()->back();
        }
    }
    //----------------------------------------------------------

    /**
     * Reset password via email
     *
     * @param Request $request
     * @return void
     */
    /*public function forgot(Request $request)
    {
        //https://www.itsolutionstuff.com/post/laravel-custom-forgot-reset-password-exampleexample.html

        if (user_id()) {
            return redirect()->route('admin.dashboard');
        }

        if ($request->method() == 'POST') {

            $validated = $request->validate([
                'email' => ['required', 'email', 'exists:users,email'],
            ]);

            $token = Str::random(64);

            DB::table('user_password_resets')->insert([
                'email' => $request->email,
                'token' => $token,
                'created_at' => Carbon::now()
            ]);

            Mail::send('admin.auth.email', ['token' => $token, 'email' => $request->email], function ($message) use ($request) {
                $message->from('no-reply@zain-clinic.com');
                $message->to($request->email);
                $message->subject('Reset Password');
            });

            session()->flash('success', sprintf(__('admin/auth.email_sent'), $request->email));
        }


        $data['title'] = __('admin/auth.forgot_title');
        $data['error'] = '';

        echo view('admin.auth.forgot', $data);
    }*/

    /**
     * Password reset via SMS
     * This function automatically sends a new password to the user
     * Not a verification code
     * @param Request $request
     * @return void
     */
    public function forgot(Request $request)
    {

        if (user_id()) {
            return redirect()->route('admin.dashboard');
        }

        if ( $request->method() == 'POST' ) {

            $validated = $request->validate([
                'mobile' => ['required', new Mobile, 'exists:users,mobile'],
            ]);

            // check if user is blocked
            $user = User::where('mobile',$request->input('mobile'))->first();
            
            if ( $user && $user->is_blocked == false ){
                
                $new_password = rand_password(10,4);

                // send sms
                $message = sprintf(__('sms.messages.reset'), $new_password);
                                
                $sent = send_sms($request->input('mobile'),$message,true,false);

                if ( $sent ){
                    // update new password AND remove the device token to logout from the mobile app
                    User::where('mobile', $validated['mobile'])->update(['password' => Hash::make($new_password)]);

                    UserActivityLog::create([
                        'user_id' => user_id(),
                        'route' => $request->route()->getName(),
                        'request_method' => $request->method(),
                    ]);
                }  

                return redirect()->route('admin.auth.login')->with('success', 'New password has been sent to your mobile.');
            
            }else{
                session()->flash('error','Unknown mobile number.');
            }

        }

        $data['title'] = __('admin/auth.forgot_title');
        $data['error'] = '';

        echo view('admin.auth.forgot', $data);
    }
    

    public function resetPass(Request $request)
    {
        if (user_id()) {
            return redirect()->route('admin.dashboard');
        }

        if ($request->method() == 'POST') {

            $validated = $request->validate([
                'password' => ['required', 'min:6', 'max:20'],
                'cpassword' => ['required', 'same:password'],
                'email' => ['required', 'email', 'exists:users,email'],
                'token' => ['required'],
            ]);

            $resetRow = DB::table('user_password_resets')
                ->where(['email' => $request->email, 'token' => $request->token])->first();

            if (!$resetRow) {
                return back()->withInput()->with('error', 'Invalid!');
            }

            $user = User::where('email', $request->email)
                ->update(['password' => Hash::make($request->password)]);

            DB::table('user_password_resets')->where(['email' => $request->email])->delete();

            return redirect()->route('admin.auth.login')->with('success', __('admin/auth.changed'));
        }

        $data['email'] = $request->input('email');
        $data['token'] = $request->input('token');
        $data['title'] = __('admin/auth.forgot_title');
        $data['error'] = '';

        echo view('admin.auth.reset', $data);
    }
}
