<?php

namespace App\Http\Controllers\Admin;

use App\Libraries\Tabler;
use App\Models\Branch;
use App\Models\Region;
use App\Rules\ArabicAlphaNumericSpaceDash;
use App\Rules\ArEnAlphaNumSpace;
use App\Rules\GeneralText;
use App\Rules\Mobile;
use App\Rules\Phone;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class BranchesController extends Controller
{
    public function index(Request $request){

        $table = $this->getBranchesTable($request);
        $data = $table->initTable();
        
        $this->loadView('branches.index','branches.title','branches',[],$data);

    }

    public function add(Request $request){

        if ( $request->getMethod() == 'POST'){

            $validated = $request->validate([
                'branch'  => ['required','min:3', 'max:200', new ArabicAlphaNumericSpaceDash],
                'region' => ['required', 'exists:regions,id'],
                'phone' => ['nullable', 'unique:branches,phone',new Phone],
                'mobile' => ['nullable', 'unique:branches,mobile',new Mobile],
                'address' => ['nullable', new GeneralText],
                'start_time' => ['required','date_format:H:i','before:end_time'],
                'end_time' => ['required','date_format:H:i','after:start_time'],
            ]);

            $insert = [
                'branch' => $validated['branch'],
                'region_id' => $validated['region'],
                'phone' => $validated['phone'],
                'address' => $validated['address'],
                'work_start_time' => date('H:i',strtotime($validated['start_time'])),
                'work_end_time' => date('H:i',strtotime($validated['end_time'])),
            ];

            Branch::create($insert);

            return redirect()->route('admin.branches')->with('success', __('admin/common.msgs.success.add'));

        }

        $data['regions'] = Region::orderBy('region')->get();

        $this->loadView('branches.add','branches.add_title','branches.add',[],$data);
    }


    public function edit(Request $request,Branch $branch){

        if ( $request->getMethod() == 'POST'){

            $validated = $request->validate([
                'branch' => ['required', Rule::unique('branches', 'branch')->ignore($branch->id), 'min:3', 'max:200', new ArEnAlphaNumSpace],
                'region' => ['required', 'exists:regions,id'],
                'phone' => ['nullable', Rule::unique('branches', 'phone')->ignore($branch->id),new Phone],
                'mobile' => ['nullable', Rule::unique('branches', 'mobile')->ignore($branch->id),new Mobile],
                'address' => ['nullable', new GeneralText],
                'start_time' => ['required','date_format:H:i','before:end_time'],
                'end_time' => ['required','date_format:H:i','after:start_time'],
            ]);

            $update = [
                'branch' => $validated['branch'],
                'region_id' => $validated['region'],
                'phone' => $validated['phone'],
                'address' => $validated['address'],
                'work_start_time' => date('H:i',strtotime($validated['start_time'])),
                'work_end_time' => date('H:i',strtotime($validated['end_time'])),
            ];

            Branch::where('id', $branch->id)->update($update);

            return redirect()->route('admin.branches')->with('success', __('admin/common.msgs.success.edit'));
        }

        $data['regions'] = Region::orderBy('region')->get();
        $data['branch'] = $branch;

        $this->loadView('branches.edit','branches.edit_title','branches.edit',$branch,$data);
    }

    //------------------------------------------------
    public function delete(Request $request, $branch_id){
        
        // branch info
        $branch = Branch::findOrFail($branch_id);

        // If branch has treatments
        $treatments = $branch->treatments;

        if ( $treatments && $treatments->count() > 0){
            return redirect()->route('admin.branches')->with('danger', __('admin/branches.delete_error.treatments'));
        }

        // If branch has appointments
        $appointments = $branch->appointments;

        if ( $appointments && $appointments->count() > 0){
            return redirect()->route('admin.branches')->with('danger', __('admin/branches.delete_error.appointments'));
        }

        // If branch has users
        $users = $branch->users;

        if ( $users && $users->count() > 0){
            return redirect()->route('admin.branches')->with('danger', __('admin/branches.delete_error.users'));
        }

        $branch->delete();

        return redirect()->route('admin.branches')->with('success', __('admin/common.msgs.success.delete'));
    }


    /*==================================================*/
    private function getBranchesTable(Request $request){

        $model = new Branch();

        $model = $model->join('regions', 'branches.region_id', '=', 'regions.id');

        $model = $model->allowedBranches();

        $th_columns = ['branch','phone','mobile','region','address','start','end'];
        $select_columns = [
            'branches.id',
            'branch',
            'phone',
            'mobile',
            'region',
            'address',
            DB::raw('DATE_FORMAT(`work_start_time`, "%h:%i %p") as start'),
            DB::raw('DATE_FORMAT(`work_end_time`, "%h:%i %p") as end'),
        ];
        $sortable_columns = ['branch'];
        $searchable_columns = ['branch'];

        $route_key = 'admin.branches';

        $table = new Tabler($route_key, $model, $select_columns, $th_columns, $sortable_columns, $searchable_columns, $request);

        return $table;
    }

    /*=======================================================*/
    public function print(Request $request, $selection)
    {
        if ($selection == 'all') {

            $select = [
                'branches.id',
                'branch',
                'phone',
                'mobile',
                'region',
                'address',
                DB::raw('DATE_FORMAT(`work_start_time`, "%h:%i %p") as start'),
                DB::raw('DATE_FORMAT(`work_end_time`, "%h:%i %p") as end'),
            ];
    
            $obj = Branch::allowedBranches()->select($select)
                    ->join('regions', 'branches.region_id', '=', 'regions.id')
                    ->orderBy('branches.branch');
                    
            $th = ['branch','phone','mobile','region','address','start','end'];

            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {
                $data['columns'][$db_col] = '<th>' . __('admin/branches.columns.' . $db_col) . '</th>';
            }

        } else {

            $table = $this->getBranchesTable($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/branches.title');

        echo view('admin.printables.list', $data);
    }



}
