<?php

namespace App\Http\Controllers\Admin;

use App\Libraries\Tabler;
use App\Models\Offer;
use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class OffersController extends Controller
{

    public function index(Request $request)
    {
        // Route Key
        $route_key = 'admin.offers';

        // Prep the model
        $model = new Offer();

        $model = $model->leftJoin('users', 'offers.created_by', '=', 'users.id');

        $user_name = DB::raw("CONCAT(users.first_name,' ',users.last_name) AS user_name");

        // Setting the parameters
        $select = ['offers.id', 'title', 'description', 'image', 'start_date', 'end_date', $user_name];

        $th = ['title', 'start_date', 'end_date', 'user_name'];

        $sortable = ['title', 'start_date', 'end_date', 'user_name'];

        //$searchable = ['first_name','second_name','third_name', 'patients.last_name','mobile','region'];
        $searchable = ['title', 'start_date', 'end_date', 'description', 'user_name'];

        $table = new Tabler($route_key, $model, $select, $th, $sortable, $searchable, $request, true);

        $data = $table->initTable();

        $this->loadView('offers.index', 'offers.title', 'offers', [], $data);
    }

    //--------------------------------------------------------------

    public function add(Request $request)
    {
        $upload_path = $this->uploadPath();

        if ($request->method() == 'POST') {

            $min_date = date('Y-m-d');
            $max_date = date('Y-m-d', strtotime("+1 month"));
            $end_min_date = date('Y-m-d', strtotime("+1 day"));
            $end_max_date = date('Y-m-d', strtotime("+6 months"));

            $validated = $request->validate([
                'title' => ['required', new GeneralText],
                'description' => ['required', new GeneralText],
                'start-date' => ['required', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date],
                'end-date' => ['required', 'before_or_equal:' . $end_max_date, 'after_or_equal:' . $end_min_date],
                'image' => ['nullable', 'mimes:png,jpg,jpeg', 'min:10', 'max:2048']
            ]);


            $offer = [
                'title' => $validated['title'],
                'description' => $validated['description'],
                'start_date' => date('Y-m-d', strtotime($validated['start-date'])),
                'end_date' => date('Y-m-d', strtotime($validated['end-date'])),
                'created_by' => user_id(),
            ];

            if ($request->file('image') && $request->file('image')->isValid()) {

                $file_name = time() . '.' . $request->file('image')->guessExtension();

                $request->file('image')->move($upload_path, $file_name);

                $offer['image'] = $file_name;
            }

            Offer::create($offer);

            return redirect()->route('admin.offers')->with('success', __('admin/common.msgs.success.add'));
        }

        $css = ['bs-datepicker/dp-bs-4.min.css'];
        $js = ['momentjs/moment.min.js', 'bs-datepicker/dp-bs-4.min.js'];

        $this->loadView('offers.add', 'offers.add_title', 'offers.add', [], [], $css, $js);
    }

    //--------------------------------------------------------------

    public function edit(Request $request, Offer $offer)
    {
        $upload_path = $this->uploadPath();

        if ($request->method() == 'POST') {

            $min_date = date('Y-m-d');
            $max_date = date('Y-m-d', strtotime("+1 month"));
            $end_min_date = date('Y-m-d', strtotime("+1 day"));
            $end_max_date = date('Y-m-d', strtotime("+6 months"));

            $validated = $request->validate([
                'title' => ['required', new GeneralText],
                'description' => ['required', new GeneralText],
                'start-date' => ['required', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date],
                'end-date' => ['required', 'before_or_equal:' . $end_max_date, 'after_or_equal:' . $end_min_date],
                'image' => ['nullable', 'mimes:png,jpg,jpeg', 'min:10', 'max:2048']
            ]);

            $update_offer = [
                'title' => $validated['title'],
                'description' => $validated['description'],
                'start_date' => date('Y-m-d', strtotime($validated['start-date'])),
                'end_date' => date('Y-m-d', strtotime($validated['end-date'])),
                'created_by' => user_id(),
            ];


            if (empty($request->input('no-image'))) {

                $update_offer['image'] = null;

                $this->deleteFile($offer->image);

            }

            if ($request->file('image') && $request->file('image')->isValid()) {

                $file_name = time() . '.' . $request->file('image')->guessExtension();

               $this->deleteFile($offer->image);

                $request->file('image')->move($upload_path, $file_name);

                $update_offer['image'] = $file_name;
            }

            $offer->update($update_offer);

            return redirect()->route('admin.offers')->with('success', __('admin/common.msgs.success.edit'));
        }

        $data['offer'] = $offer;

        $css = ['bs-datepicker/dp-bs-4.min.css'];
        $js = ['momentjs/moment.min.js', 'bs-datepicker/dp-bs-4.min.js'];

        $this->loadView('offers.edit', 'offers.edit_title', 'offers.edit', $offer, $data, $css, $js);
    }

    public function delete(Request $request, Offer $offer)
    {

        $this->deleteFile($offer->image);

        $offer->delete();

        return redirect()->route('admin.offers')->with('success', __('admin/common.msgs.success.delete'));
    }


    private function uploadPath()
    {
        // When deployed
        $upload_path = base_path('../') . '/' . config('admin.public_uploads_dir');

        // Localhost for testing
        //$upload_path = public_path(config('admin.public_uploads_dir'));

        return $upload_path;
    }

    private function deleteFile($file)
    {

        $upload_path = $this->uploadPath();

        // remove old image
        if ($file && file_exists($upload_path . '/' . $file)) {
            unlink($upload_path . '/' . $file);
        }
    }
}
