<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\Controller;
use App\Libraries\Tabler;
use App\Models\Permission;
use App\Models\Role;
use App\Rules\AlphaSpace;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class RolesController extends Controller
{
    public function index(Request $request){
        
        $model = new Role();
        $model = $model->orderBy('role');
        $route_key = 'admin.roles';
        $select = ['id','role'];
        $th_col = ['role'];
        $sortable = ['role'];
        $searchable = ['role'];

        $table = new Tabler($route_key,$model,$select,$th_col,$sortable,$searchable,$request);

        $data = $table->initTable();

        $this->loadView('roles.index','roles.title','roles',[],$data);
    }

    /**
     * dd new role
     *
     * @param Request $request
     * @return void
     */
    public function add(Request $request){

        if ( $request->getMethod() == 'POST' ){

            $validated = $request->validate([
                'role' => ['required','unique:roles,role','min:2','max:200', new AlphaSpace],
                'permissions' => ['required'],
                'permissions.*' => ['integer', 'exists:permissions,id'],
                'dashboard' => ['required','gte:0','lte:3'],
            ]);

            $role = Role::create([
                'role' => $validated['role'],
                'dashboard' => $validated['dashboard'],
            ]);

            $permissions = array_merge($validated['permissions'],$this->getGrantedPermissions());

            $role->permissions()->attach($permissions);
            
            return redirect()->route('admin.roles')->with('success', __('admin/common.msgs.success.add'));
            
        }


        $data['permissions'] = Permission::where('strict_granted',0)->get()->groupBy('controller_lang_key');

        $this->loadView('roles.add','roles.add_title','roles.add',[],$data);

    }

    /**
     * Edit existing role
     *
     * @param Request $request
     * @param Role $role
     * @return void
     */
    public function edit(Request $request, Role $role){
        
        if ($request->method() == 'POST') {

            $validated = $request->validate([
                'role' => ['required', Rule::unique('roles', 'role')->ignore($role->id), 'min:3', 'max:200', new AlphaSpace],
                'permissions' => ['required'],
                'permissions.*' => ['integer', 'exists:permissions,id'],
                'dashboard' => ['required','gte:0','lte:3'],
            ]);

            /*****************************************************************
             *  When validation fails, Laravel automatically exits the code
             * ***************************************************************/
            Role::where('id', $role->id)->update([
                'role' => $validated['role'],
                'dashboard' => $validated['dashboard'],
            ]);

            $permissions = array_merge($validated['permissions'],$this->getGrantedPermissions());

            $role->permissions()->sync($permissions);

            return redirect()->route('admin.roles')->with('success', __('admin/common.msgs.success.edit'));
        }

        $data['role'] = $role;
        
        $role_permissions = $role->permissions;

        $data['role_permissions'] = [];
        
        foreach ( $role_permissions as $permission ) {
            $data['role_permissions'][] = $permission->id;
        }

        $data['permissions'] = Permission::where('strict_granted',0)->get()->groupBy('controller_lang_key');

        $this->loadView('roles.edit', 'roles.edit_title', 'roles.edit', $role, $data);
    }

    private function getGrantedPermissions(){

        $granted = Permission::select('id')->where('strict_granted',1)->get();

        $result = []; 

        if ( $granted ){

            foreach ( $granted as $permission ){

                $result[] = $permission->id;
            }
        }

        return $result;
    }


    public function delete(Request $request, Role $role){

        if ( $role->users && $role->users->count() > 0 ){
            
            return redirect()->route('admin.roles')->with('danger', __('admin/common.msgs.error.delete'));

        }

        $role->delete();

        return redirect()->route('admin.roles')->with('success', __('admin/common.msgs.success.delete') );

    }
}
