<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Libraries\PatientCommon;
use App\Models\Patient;
use App\Rules\Mobile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;
use Illuminate\Support\Str;

class Patients extends Controller
{
    /**
     * Patient Login
     *
     * @param Request $request
     * @return void
     */
    public function login(Request $request)
    {

        if ($request->getMethod() == 'POST') {

            $patient_common = new PatientCommon();

            $result = $patient_common->login($request, true);

            return $patient_common->getResponse();
        }

        return response()->json(['error' => ['Invalid request']], 400);
    }
    //==================================================
    public function register(Request $request)
    {

        if ($request->getMethod() == 'POST') {

            $patient_common = new PatientCommon();

            $result = $patient_common->register($request, true, true);

            return $patient_common->getResponse();
        }

        return response()->json(['error' => ['Invalid request']], 400);
    }
    //==================================================
    public function update(Request $request, $patient_id)
    {

        $user_id = $this->getUserByToken($request);
        //--- if the sent id doesn't match the id from db, 403 forbidden 

        if ($patient_id != $user_id) {
            return response()->json(['error' => '403'], 403);
        }

        if ($request->getMethod() == 'POST') {

            $patient_common = new PatientCommon();

            $patient = Patient::findOrFail($patient_id);

            $result = $patient_common->updateProfile($request, $patient, true, true);

            return $patient_common->getResponse();
        }

        return response()->json(['error' => ['Invalid request']], 400);
    }
    //==================================================
    public function validateUpdate(Request $request, $patient_id)
    {

        $user_id = $this->getUserByToken($request);
        //--- if the sent id doesn't match the id from db, 403 forbidden 

        if ($patient_id != $user_id) {
            return response()->json(['error' => '403'], 403);
        }

        if ($request->getMethod() == 'POST') {

            $patient_common = new PatientCommon();

            $patient = Patient::findOrFail($patient_id);

            $result = $patient_common->updateProfile($request, $patient, true, true, false);

            return $patient_common->getResponse();
        }

        return response()->json(['error' => ['Invalid request']], 400);
    }
    //==================================================
    public function verify(Request $request)
    {

        if ($request->method() == 'POST') {

            $rules = [
                'code' => ['required', 'integer', 'digits:6'],
                'mobile' => ['required', new Mobile, 'exists:patients,mobile']
            ];

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {

                return response()->json(['error' => $validator->errors()], 400);
            }

            $db_otp_row = DB::table('patient_otp')->where('mobile', $request->input('mobile'))->first();

            if ($db_otp_row) {

                if (Hash::check($request->input('code'), $db_otp_row->code)) {

                    Patient::where('mobile', $request->input('mobile'))->update(['is_verified' => 1]);

                    DB::table('patient_otp')->where('mobile', $request->input('mobile'))->delete();

                    return response()->json(['success' => 'success'], 200);
                } else {
                    return response()->json(['error' => 'Invalid code'], 400);
                }
            } else {
                return response()->json(['error' => 'Unknown error occurred'], 400);
            }
        }

        return response()->json(['error' => ['Invalid request']], 400);
    }
    //==================================================
    public function resetPassword(Request $request)
    {

        if ($request->method() == 'POST') {

            $rules = [
                'mobile' => ['required', new Mobile, 'exists:patietns,mobile']
            ];

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {

                return response()->json(['error' => $validator->errors()], 400);
            }

            $new_password = Str::random(10);

            // send sms
            $message = sprintf(__('sms.messages.reset'), $new_password);

            $sms_result = send_sms($request->input('mobile'), $message);

            if ($sms_result == 'success') {
                // update new password AND remove the device token to logout from the mobile app
                Patient::where('mobile', $request->input('mobile'))->update(['password' => Hash::make($new_password), 'device_token' => null]);

                return response()->json(['success' => 'success'], 200);
            } else {
                return response()->json(['error' => $sms_result], 400);
            }
        }

        return response()->json(['error' => ['Invalid request']], 400);
    }
    //==================================================
    public function logout(Request $request, $patient_id)
    {

        // prevent the user from seeing other patients
        // info in case of token hijacking        

        $user_id = $this->getUserByToken($request);
        //--- if the sent id doesn't match the id from db, 403 forbidden 

        if ($patient_id != $user_id) {
            return response()->json(['error' => '403'], 403);
        }

        $patient = Patient::findOrFail($patient_id);

        $patient->tokens()->delete();

        return response()->json(['success' => 'success'], 200);
    }
    //==================================================
    /**
     * Get Patient details
     *
     * @param int $id
     * @return void
     */
    public function getPatient(Request $request, $id)
    {
        // prevent the user from seeing other patients
        // info in case of token hijacking        

        $user_id = $this->getUserByToken($request);
        //--- if the sent id doesn't match the id from db, 403 forbidden 

        if ($id != $user_id) {
            return response()->json(['error' => '403'], 403);
        }

        $patient = Patient::findOrFail($id);

        $patient_info = [
            'id' => $patient->id,
            'first_name' => $patient->first_name,
            'second_name' => $patient->second_name,
            'third_name' => $patient->third_name,
            'last_name' => $patient->last_name,
            'dob' => $patient->formatted_dob,
            'gender' => $patient->full_gender,
            'region' => $patient->region->region,
            'address' => $patient->address,
            'mobile' => $patient->mobile,
            'email' => $patient->email,
        ];

        return response()->json(['patient_info' => $patient_info], 200);
    }

    //==================================================

    private function getUserByToken(Request $request)
    {
        //--- Get the token from request
        $header = $request->header('authorization');

        $token = str_replace('|', '', strstr($header, '|'));

        //--- Get the user id should be using this token
        $hashedToken = hash('sha256', $token);

        $db_token = PersonalAccessToken::where('token', $hashedToken)->first();

        $user = $db_token->tokenable;

        return $user->id;
    }

    //-----------------------------------
    public function setDeviceToken(Request $request)
    {

        $rules = [
            'patient_id' => ['required', 'exists:patients,id'],
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {

            return response()->json(['error' => '400'], 400);
        }

        $firebase_token = $request->input('device_token');
        $patient_id = $request->input('patient_id');

        // get patient
        Patient::where('id', $patient_id)->update(['device_token' => $firebase_token]);

        return response()->json(['success' => 'success'], 200);
    }
}
