<?php

namespace App\Libraries;

use App\Models\Appointment;
use App\Models\Patient;
use App\Models\Treatment;
use App\Models\Branch;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

class AppointmentsCommon
{

    private $is_web_request;

    private $response;

    public function __construct($is_web = true)
    {
        $this->is_web_request = $is_web;
        $this->response = '';
    }

    //---------------------------------------------------------------
    /**
     * return the response (if errors)
     *
     * @return string
     */
    public function getResponse()
    {
        return $this->response;
    }
    //---------------------------------------------------------------
    public function getPatientAppointments(Request $request, $patient_id)
    {

        $this->response = '';

        $model = new Appointment();

        $select = ['ref_number', 'appointments.id as id', 'appointment_date', 'start_time', 'treatments.treatment AS treatment', 'branches.branch as branch'];

        $model = $model->join('treatments', 'appointments.treatment_id', '=', 'treatments.id');

        $model = $model->join('branches', 'appointments.branch_id', '=', 'branches.id');

        $model = $model->join('patients', 'appointments.patient_id', '=', 'patients.id');

        $model = $model->where('appointments.postponed', '=', 0);

        $model = $model->where('appointments.patient_id', '=', $patient_id);

        $model = $model->orderBy('appointment_date')->orderBy('start_time');

        $th = ['ref_number', 'appointment_date', 'treatment', 'branch', 'start_time'];

        $sortable = ['ref_number', 'appointment_date', 'treatment', 'branch', 'start_time'];

        $searchable = ['ref_number', 'appointment_date', 'treatment', 'branch'];

        $table = new Tabler('admin.appointments', $model, $select, $th, $sortable, $searchable, $request, true);

        $data = $table->initTable();

        return view('admin.appointments.index', $data);
    }
    //---------------------------------------------------------------
    public function addAppointment(Request $request)
    {
        $this->response = '';

        $min_date = date('Y-m-d');
        $max_date = date('Y-m-d', strtotime('+13 weeks'));

        $rules = [
            'ap-date' => ['required', 'date_format:Y-m-d', 'after_or_equal:' . $min_date, 'before_or_equal:' . $max_date],
            'ap-session' => ['required', 'date_format:h:i A'],
        ];
        // Filter addition by branch for web admin users
        if ($this->is_web_request && user_id()) {
            $rules['patient'] = ['required', 'exists:patients,id', Rule::in(allowed_patients())];
            $rules['branch'] = ['required', 'exists:branches,id', Rule::in(allowed_branches())];
            $rules['treatment'] = [
                'required',
                Rule::exists('branch_treatment', 'treatment_id')
                    ->where('branch_id', $request->input('branch'))
                    ->whereIn('branch_id', allowed_branches())
            ];
        } else {
            $rules['patient'] = ['required', 'exists:patients,id'];
            $rules['branch'] = ['required', 'exists:branches,id'];
            $rules['treatment'] = ['required', Rule::exists('branch_treatment', 'treatment_id')->where('branch_id', $request->input('branch'))];
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {

            $errorString = implode("<br>", $validator->errors()->all());
            $this->response = response()->json(['error' => $errorString], 400);
            return;
        }

        // Logical validation
        $patient_id = $request->input('patient');
        $branch_id = $request->input('branch');
        $treatment_id = $request->input('treatment');
        $app_date = $request->input('ap-date');
        $app_session = date('H:i', strtotime($request->input('ap-session')));

        $result = $this->validateNewAppointment($patient_id, $treatment_id, $branch_id, $app_date, $app_session);

        // if array has an error
        if (empty($this->response)) {

            // check if the appointments ref_number exists, update the record
            // by setting the deleted flag to null

            $app_exists = Appointment::withTrashed()->where('patient_id', $patient_id)->where('ref_number', $result['ref_number'])->first();

            if ($app_exists) {

                $app_exists->restore();

                $new_app = true;
            } else {
                $result['notes'] = $request->input('appointment-note');
                $new_app = Appointment::create($result);
            }

            // if appointment added successfully
            if ($new_app) {

                if ($this->is_web_request) {
                    session()->flash('success', __('admin/common.msgs.success.add'));
                }

                $this->response = response()->json(['success' => 'success']);
            } else {

                $this->response = response()->json(['error' => '510'], 500);
            }
        }
    }
    //---------------------------------------------------------------

    public function changeAppointment(Request $request, Appointment $appointment)
    {
        $this->response = '';

        $min_date = date('Y-m-d');
        $max_date = date('Y-m-d', strtotime('+13 weeks'));

        // If changing past appointment
        $app_date_before_update = $appointment->appointment_date;
        $app_start_time_before_update = date('H:i', strtotime($appointment->start_time));
        $today = date('Y-m-d');
        $time = date('H:i');

        if ($app_date_before_update < $today || ($app_date_before_update == $today && $app_start_time_before_update < $time)) {
            $this->response = response()->json(['error' => __('admin/appointments.error.passed_app')], 400);
            return;
        }

        $rules = [
            'patient' => ['required', 'exists:patients,id'],
            'treatment' => ['required', Rule::exists('branch_treatment', 'treatment_id')->where('branch_id', $request->input('branch'))],
            'ap-date' => ['required', 'date_format:Y-m-d', 'after_or_equal:' . $min_date, 'before_or_equal:' . $max_date],
            'ap-session' => ['required', 'date_format:h:i A'],
        ];

        // Filter addition by branch for web admin users
        if ($this->is_web_request && user_id()) {
            $rules['branch'] = ['required', 'exists:branches,id', Rule::in(allowed_branches())];
        } else {
            $rules['branch'] = ['required', 'exists:branches,id'];
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $errorString = implode("<br>", $validator->errors()->all());
            $this->response = response()->json(['error' => $errorString], 400, [], JSON_UNESCAPED_UNICODE);
            return;
        }

        // Logical validation
        $patient_id = $request->input('patient');
        $branch_id = $request->input('branch');
        $treatment_id = $request->input('treatment');
        $app_date = $request->input('ap-date');
        $app_session = date('H:i', strtotime($request->input('ap-session')));


        // check if anything changed in the appointment or saved without modification
        // if not, redirect and return a success message WITHOUT updating the database
        if (
            $appointment->patient_id != $patient_id || $appointment->branch_id != $branch_id ||
            $appointment->treatment_id != $treatment_id || $appointment->appointment_date != $app_date ||
            date('H:i', strtotime($appointment->start_time)) != $app_session
        ) {

            // validate & do the update
            //--- First: set the current appointment postponed to 1 temporarily ( will be changed to the new record later)
            Appointment::where('id', $appointment->id)->update(['postponed' => 1]);

            $result = $this->validateNewAppointment($patient_id, $treatment_id, $branch_id, $app_date, $app_session);

            // if the return object is json => then it has an error
            // if array has an error
            if (empty($this->response)) {

                //--- Second: add the new appointment
                //$app = Appointment::create($result);
                // New logic => no postpone, update directly
                $result['notes'] = $request->input('appointment-note');
                Appointment::where('id', $appointment->id)->update($result);

                // check if the appointments ref_number exists, update the record
                // by setting the deleted flag to null

                //Appointment::where('id', $appointment->id)->update(['postponed' => 0]);

                if ($this->is_web_request) {

                    session()->flash('success', __('admin/common.msgs.success.edit'));
                }
            }
            // update the old record and set the correct postponed id
            $result['notes'] = $request->input('appointment-note');
            Appointment::where('id', $appointment->id)->update(['postponed' => 0]);
        }

        if (empty($this->response)) {
            $this->response = response()->json(['success' => 'success'], 200);
        } else {
            return $this->response;
        }



        return;
    }
    //---------------------------------------------------------------
    public function generateCalendar($patient_id, $branch_id, $treatment_id, $week_number, $scope = 'admin')
    {

        $treatment = Treatment::findOrFail($treatment_id);

        //------------------------
        $treatment_devices = $this->getTreatmentsDevices($treatment_id, $branch_id);

        // get ALL typical sessions (without filtering)
        $all_sessions = $this->getAllTreatmentSessions($treatment->session_period, $branch_id);

        // get the selected week dates
        $week_dates = $this->getWeekDates($week_number);

        // initialize the calendar array
        $calendar = [];

        // prep the loop
        $current_date = date('Y-m-d');
        $current_time = date('H:i');
        // to prevent the patient from booking an immediate appointment
        // must be at least after one hour
        $allowed_booking_time = date('H:i', strtotime($current_time . '+30 minutes'));

        // loop through every date in the week to filter the sessions
        foreach ($week_dates as $date => $day) {
            $free_sessions = [];

            foreach ($all_sessions as $session) {

                // if week date == current date and session time is in the past, prevent
                // patient from booking the session
                if ($current_date == $date && $session <= $allowed_booking_time) {
                    continue;
                }

                // get the taken appointments for the selected branch, treatment, date, and start time
                $appointments = $this->getTreatmentAppointments(null, $branch_id, $treatment_id, $date, $session);

                if ($appointments->count() < $treatment_devices) {

                    // Here means there is an empty device for new appointment
                    // Now, lets check if the patient does have another appointment on 
                    //the same date for another treatment in the same branch

                    $apps_for_another_treatment = $this->getTreatmentAppointments($patient_id, $branch_id, null, $date);

                    // Loop through other appointments for different treatments for the same patient
                    // in the same branch. if one appointment conflicts with the session, break the loop.

                    $session_is_valid = true;

                    foreach ($apps_for_another_treatment as $another_app) {

                        $first_app_start = strtotime($session);
                        $first_app_end = strtotime($session . ' +' . $treatment->session_period . ' minutes');

                        $other_app_start = strtotime($another_app->start_time);
                        $other_app_end = strtotime($another_app->end_time);

                        // if the other appointment starts in the selected session or ends in the selected session,
                        // break the loop and set the flag to false.
                        if (($first_app_start >= $other_app_start &&  $first_app_start < $other_app_end) || $first_app_end >= $other_app_start &&  $first_app_end < $other_app_end) {
                            $session_is_valid = false;
                            break;
                        }
                    }
                    // if everything clears out, save the session as free
                    if ($session_is_valid) {
                        $free_sessions[] = date('h:i A', strtotime($session));
                    }
                }
            }

            $calendar[$date] = [
                'date' => $date,
                'day' => $day,
                'sessions' => $free_sessions,
            ];
        }

        // Setting the data array
        $data['calendar'] = $calendar;

        // For web use only
        if ($this->is_web_request) {
            $first_day = array_keys($week_dates)[0];
            $last_day = array_keys($week_dates)[5];

            // set the next,prev links to navigate through weeks
            if ($week_number == 0) {

                $next_week = $week_number + 1;

                $data['next_week'] = route($scope . '.appointments.get_calendar', [$patient_id, $branch_id, $treatment_id, $next_week]);
                $data['prev_week'] = '';
            } elseif ($week_number == 12) {

                $prev_week = $week_number - 1;

                $data['next_week'] = '';
                $data['prev_week'] = route($scope . '.appointments.get_calendar', [$patient_id, $branch_id, $treatment_id, $prev_week]);
            } else {

                $next_week = $week_number + 1;
                $prev_week = $week_number - 1;

                $data['next_week'] = route($scope . '.appointments.get_calendar', [$patient_id, $branch_id, $treatment_id, $next_week]);
                $data['prev_week'] = route($scope . '.appointments.get_calendar', [$patient_id, $branch_id, $treatment_id, $prev_week]);
            }

            $data['selected_date'] = '';
            $data['selected_time'] = '';

            $data['date_range_title'] = date('d/m/Y', strtotime($first_day)) . '     -     ' . date('d/m/Y', strtotime($last_day));
        }

        return $data;
    }
    //---------------------------------------------------------------
    private function getTreatmentsDevices($treatment_id, $branch_id)
    {
        try {

            /*  if ( $this->is_web_request && user_id() && !in_array($branch_id,allowed_branches() ) ){
            
                $this->response = response()->json(['error' => '404 Branch Not Found']);
            
            } */

            $treatment = Branch::with(['treatments' => function ($q) use ($treatment_id) {
                $q->where('id', $treatment_id)->first();
            }])->findOrFail($branch_id)->treatments;

            if ($treatment->count() != 1) {
                return 0;
            }

            // The selected treatment single session duration
            return (int) $treatment->first()->pivot->devices;
        } catch (Exception $e) {

            Log::error($e);
            $this->response = response()->json(['error' => '500 Exception']);
        }
    }

    //--------------------------------------------------------------
    private function getAllTreatmentSessions($duration = null, $branch_id)
    {

        $treatment_duration = (!$duration) ? (int) config('admin.min_session_duration') : $duration;

        // get branch working hours
        $branch = Branch::findOrFail($branch_id);

        $to = Carbon::parse($branch->end_time);
        $from = Carbon::parse($branch->start_time);

        //$work_day_in_minutes = (int)config('admin.work_hours');

        // Make sure the diff is mor than 300 minutes (5 hours)
        $work_day_in_minutes = ((int) $to->diffInMinutes($from) > 300) ? (int) $to->diffInMinutes($from) : (int)config('admin.work_hours');

        //$work_start_time = config('admin.start_time');
        $work_start_time = $branch->start_time;

        $all_sessions = [];

        $no_of_sessions = $work_day_in_minutes / $treatment_duration;

        for ($i = 0; $i < $no_of_sessions; $i++) {

            $shift = $i * $treatment_duration;

            $all_sessions[] =  date('H:i:s', strtotime('+' . $shift . ' minutes', strtotime($work_start_time)));
        }

        // validate the last session if valid before work day ends (before 5:00 PM) or not 

        $work_end_time = strtotime($work_start_time . ' +' . $work_day_in_minutes . ' minutes');

        $last_session = end($all_sessions);

        $diff = ($work_end_time - strtotime($last_session)) / 60;

        if ($diff < $treatment_duration) {
            array_pop($all_sessions);
        }

        return $all_sessions;
    }

    //--------------------------------------------------------------
    private function getWeekDates($week_number)
    {
        // If no week number is specified, get apps for the whole month
        // used for the mobile app
        if ($week_number == -1) {

            $first_day = date('Y-m-d');

            $max_days_count = 31;
        } else {

            // used for web admin side 
            $first_day = date('Y-m-d', strtotime('+' . $week_number . ' weeks'));

            $max_days_count = 6;
        }

        $day_count = 0;

        $date = $first_day; // starting value

        $result = [];

        while ($day_count < $max_days_count) {

            $day = date('l', strtotime($date)); // get the day name

            if (!in_array($day, config('admin.off_days'))) {

                $result[$date] = $day;

                $day_count++;
            }

            $date = date('Y-m-d', strtotime($date . " +1 days"));
        }

        return $result;
    }

    //--------------------------------------------------------------
    private function getTreatmentAppointments($patient_id = null, $branch_id = null, $treatment_id = null, $date = null, $session = null)
    {

        $model = new Appointment();

        if ($patient_id) {

            $model = $model->where('patient_id', $patient_id);
        }

        if ($branch_id) {

            $model = $model->where('branch_id', $branch_id);
        }

        if ($treatment_id) {

            $model = $model->where('treatment_id', $treatment_id);
        }

        if ($date) {

            $model = $model->where('appointment_date', $date);
        }

        if ($session) {

            $model = $model->where('start_time', $session);
        }

        // Ignore postponed records
        $model = $model->where('postponed', 0)->where('scheduled', 1);

        return $model->get();
    }

    //--------------------------------------------------------------
    /**
     * Validate appointment booking (add, postpone)
     *
     * @param int $patient_id
     * @param int $treatment_id
     * @param int $branch_id
     * @param int $app_date
     * @param int $app_session
     * @return array|void
     */
    public function validateNewAppointment($patient_id, $treatment_id, $branch_id, $app_date, $app_session)
    {
        $this->response = '';

        $treatment = Treatment::findOrFail($treatment_id);
        $treatment_devices = $this->getTreatmentsDevices($treatment_id, $branch_id);

        if (!empty($this->response)) {
            return;
        }

        $start_time = date('H:i', strtotime($app_session));
        $end_time = date('H:i', strtotime($start_time . '+' . $treatment->session_period . ' minutes'));


        //-- If the selected session is within working hours
        $branch = Branch::findOrFail($branch_id);

        $branch_start_time = $branch->start_time;
        $branch_end_time = $branch->end_time;

        if ($start_time < $branch_start_time ||  $end_time > $branch_end_time) {

            $this->response = response()->json(['error' => __('admin/appointments.error.invalid_Session')], 400);
            return;
        }


        //-- If there are any empty devices (if still an empty appointment in case another patient booked it)
        $appointments = $this->getTreatmentAppointments(null, $branch_id, $treatment_id, $app_date, $app_session);

        if ($appointments->count() >= $treatment_devices) {

            $this->response = response()->json(['error' => __('admin/appointments.error.not_empty')], 400);
            return;
        }
        // check if user has booked the same treatment twice on the same day
        $same_treatment = $this->getTreatmentAppointments($patient_id, $branch_id, $treatment_id, $app_date);

        if ($same_treatment->count()) {

            $this->response = response()->json(['error' => __('admin/appointments.error.same_treatment')], 400);

            return;
        }
        // Now, lets check if the patient does have another appointment on 
        // the same date for another treatment, in the same branch to make sure there is no overlap
        $apps_for_another_treatment = $this->getTreatmentAppointments($patient_id, $branch_id, null, $app_date);

        // Loop through other appointments for different treatments for the same patient
        // in the same branch. if one appointment conflicts with the session, break the loop.
        foreach ($apps_for_another_treatment as $another_app) {

            $first_app_start = strtotime($app_session);
            $first_app_end = strtotime($app_session . ' +' . $treatment->session_period . ' minutes');

            $other_app_start = strtotime($another_app->start_time);
            $other_app_end = strtotime($another_app->start_time);

            // if the other appointment starts in the selected session or ends in the selected session,
            // break the loop and set the flag to false.
            if (($other_app_start >= $first_app_start &&  $other_app_start <= $first_app_end) || $other_app_end >= $first_app_start &&  $other_app_end <= $first_app_end) {

                $this->response = response()->json(['error' => __('admin/appointments.error.overlap')], 400);

                return;

                break;
            }
        }

        // Check if the patient has appointment in another branch on the same
        // date, return false with an error message
        $apps_in_other_branches = Appointment::where('patient_id', $patient_id)
            ->where('appointment_date', $app_date)
            ->where('branch_id', '!=', $branch_id)
            ->where('postponed', 0)
            ->where('scheduled', 1)
            ->get();

        if ($apps_in_other_branches->count()) {

            $this->response = response()->json(['error' => __('admin/appointments.error.app_other_branch')], 400);

            return;
        }

        //----------------------------------------------------------------------------------------
        // Insert 
        $ap_info['branch_id'] = $branch_id;
        $ap_info['treatment_id'] = $treatment_id;
        $ap_info['patient_id'] = $patient_id;
        $ap_info['appointment_date'] = $app_date;
        $ap_info['start_time'] = date('H:i', strtotime($app_session));
        $ap_info['end_time'] = $end_time;

        $ref_number = $this->genRefNumber($branch_id, $treatment_id, $patient_id, $app_session, $app_date);
        $ap_info['ref_number'] = $ref_number;

        return $ap_info;
    }
    //--------------------------------------------------------------------
    private function genRefNumber($branch_id, $treatment_id, $patient_id, $session_time, $app_date)
    {

        $year = date('y', strtotime($app_date));
        $month = date('m', strtotime($app_date));
        $day = date('d', strtotime($app_date));
        $hour = date('Hi', strtotime($session_time));

        $ref_number = $year . $month . $day . $hour . $branch_id . $treatment_id . $patient_id;

        return (string) $ref_number;
    }
}
