<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\Controller;
use App\Libraries\Tabler;
use App\Models\Complaint;
use App\Models\ComplaintComments;
use App\Models\Patient;
use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ComplaintsController extends Controller
{
    /**
     * List Complaints
     *
     * @param Request $request
     * @return void
     */
    public function index(Request $request)
    {
        $table = $this->getComplaintsTable($request);

        $data = $table->initTable();

        $this->loadView('complaints.index', 'complaints.title', 'complaints', [], $data);
    }
    //-----------------------------------------------------------

    /**
     * Add new complaint
     *
     * @param Request $request
     * @return void
     */
    public function add(Request $request)
    {

        if ($request->getMethod() == 'POST') {

            $min_date = date('Y-m-d', strtotime("-1 week"));
            $max_date = date('Y-m-d');

            $rules = [
                'description' => ['required', 'max:5000', new GeneralText],
                'complaint-date' => ['required', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date],
                'attachment' => ['mimes:png,jpg,jpeg,mp4,m4v', 'min:10', 'max:20480']
            ];

            if ( !is_admin() ){
                $rules['patient'] = ['required', 'exists:patients,id',Rule::in(allowed_patients())];
            }else{
                $rules['patient'] = ['required', 'exists:patients,id'];
            }

            $validated = $request->validate($rules);

            $complaint = [
                'patient_id' => $validated['patient'],
                'description' => $validated['description'],
                'complaint_date' => date('Y-m-d', strtotime($validated['complaint-date'])),
                'created_by' => user_id()
            ];

            if ($request->file('attachment') && $request->file('attachment')->isValid()) {

                $patient_dir = config('admin.complaints_dir') . '/' . $validated['patient'];

                $file_name = time() . '.' . $request->file('attachment')->guessExtension();

                $request->file('attachment')->storeAs($patient_dir, $file_name);

                $complaint['attachment'] = $file_name;
            }

            Complaint::create($complaint);

            return redirect()->route('admin.complaints')->with('success', __('admin/common.msgs.success.add'));
        }


        $css = ['bs-select/bootstrap-select.min.css','bs-datepicker/dp-bs-4.min.css','bs-datepicker/dp-bs-4.min.js'];
        $js = ['bs-select/bootstrap-select.min.js','momentjs/moment.min.js'];


        $data['patients'] = Patient::allowedPatients()->get();

        $this->loadView('complaints.add', 'complaints.add_title', 'complaints.add', [], $data,$css,$js);
    }
    //-------------------------------------------------------------------------
    /**
     * Complaint info
     *
     * @param Request $request
     * @param Complaint $complaint
     * @return void
     */
    public function info(Request $request, Complaint $complaint)
    {

        // get get patient info
        $data['patient'] = Complaint::find($complaint->id)->patient;

        $data['appointments'] = $data['patient']->appointments;

        $data['complaint'] = $complaint;

        $data['comments'] = $complaint->comments;

        $this->loadView('complaints.info', 'complaints.info_title', 'complaints.info', $complaint, $data);
    }
    //--------------------------------------------------------------------------
    public function resolved(Request $request, Complaint $complaint){

        $complaint->update(['is_resolved' => !(bool)$complaint->is_resolved]);

        return redirect()->route('admin.complaints')->with('success', __('admin/common.msgs.success.edit'));

    }

    //--------------------------------------------------------------------------
    /**
     * Download Complaint attachment
     *
     * @param Request $request
     * @param Complaint $complaint
     * @return void
     */
    public function downloadAttachment(Request $request, Complaint $complaint)
    {
        if ( !is_admin() && !in_array($complaint->patient_id,allowed_patients()) ){
            abort(404);
        }

        $path = config('admin.complaints_dir') . '/' . $complaint->patient_id . '/' . $complaint->attachment;

        if (Storage::exists($path)) {

            return Storage::download($path);
        }

        abort(404);
    }
    //-----------------------------------------------------------------------------

    public function addComment(Request $request)
    {

        if ( $request->method() == 'POST') {

            $rules = [
                'complaint-id' => ['required', 'exists:complaints,id'],
                'comment' => ['required', 'min:5', 'max:4000', new GeneralText],
            ];

            $validated = $request->validate($rules);


            $complaint = Complaint::findOrFail($validated['complaint-id']);
            
            if ( !is_admin() && !in_array($complaint->patient_id,allowed_patients()) ){
                abort(404);
            }

            $data['comment'] = $validated['comment'];
            $data['complaint_id'] = $validated['complaint-id'];
            $data['added_by'] = user_id();
            
            $data['comment'] = ComplaintComments::create($data);

            return redirect()->route('admin.complaints.info',$validated['complaint-id']);
        }
    }
    //-----------------------------------------------------------------------------

    public function refreshMessageBox(Request $request, int $complaint_id, int $last_msg_id = 0)
    {
        /* if ( $request->ajax() && $request->getMethod() == 'POST') {

            $messages = ComplaintComments::where('id', '>', $last_msg_id)->where('complaint_id', $complaint_id)->get();

            $complaint_messages = '';

            $new_last_id = $last_msg_id;

            if ($messages->count() > 0) {

                foreach ($messages as $message) {

                    if ($message->user_id == user_id()) {
                        $complaint_messages .= view('admin.complaints.messages.message_box_right', ['message' => $message]);
                    } else {
                        $complaint_messages .= view('admin.complaints.messages.message_box_left', ['message' => $message]);
                    }
                }

                $new_last_id = ComplaintComments::where('complaint_id', $complaint_id)->orderBy('id', 'DESC')->first()->id;
            }

            $json = [
                'last_id' => $new_last_id,
                'messages' => $complaint_messages
            ];

            return response()->json($json);
        } */
    }

    /*=======================================================*/
    public function print(Request $request, $selection)
    {
        if ($selection == 'all') {

            $model = new Complaint();
            //$model = $model->with('patients');
            $select = $model->getComplaintsSelect();
            
            $obj = Complaint::allowedPatients()->select($select)
                    ->join('patients', 'complaints.patient_id', '=', 'patients.id')
                    ->leftJoin('users', 'complaints.created_by', '=', 'users.id')
                    ->orderBy('complaint_date','DESC');
                    
            $th = ['complaint_date', 'patient_name','description', 'is_resolved', 'created_at', 'updated_at','user_name'];

            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {
                $data['columns'][$db_col] = '<th>' . __('admin/complaints.columns.' . $db_col) . '</th>';
            }

        } else {

            $table = $this->getComplaintsTable($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/complaints.title');

        echo view('admin.printables.list', $data);
    }
    /*=======================================================*/
    private function getComplaintsTable(Request $request){

        $model = new Complaint();
        //$model = $model->with('patients');
        $select = $model->getComplaintsSelect();
        
        $model = $model->allowedPatients();

        $model = $model->join('patients', 'complaints.patient_id', '=', 'patients.id');

        $model = $model->leftJoin('users', 'complaints.created_by', '=', 'users.id');
       
        $th = ['complaint_date', 'patient_name','description', 'is_resolved', 'created_at', 'updated_at','user_name'];

        $sortable = ['complaint_date','patient_name'];
        
        $searchable = ['description','patient_name','is_resolved'];

        $table = new Tabler('admin.complaints', $model, $select, $th, $sortable, $searchable, $request,true);

        return $table;
    }
}
