<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Complaint;
use App\Models\Patient;
use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Laravel\Sanctum\PersonalAccessToken;
use Illuminate\Support\Facades\Validator;

class Complaints extends Controller
{
    public function patientComplaints(Request $request, $patient_id)
    {
        $user_id = $this->getUserByToken($request);
        //--- if the sent id doesn't match the id from db, 403 forbidden 

        if ($patient_id != $user_id) {
            return response()->json(['error' => '403'], 403);
        }

        $patient = Patient::findOrFail($patient_id);

        $result = [];

        if ($patient && $patient->complaints) {
            
            foreach ($patient->complaints as $complaint){

                $result[] = [
                    'id' => $complaint->id,
                    'description' => $complaint->description,
                    'complaint_date' => date('Y-m-d', strtotime($complaint->complaint_date)),
                    'attachment' => ($complaint->attachment) ? route('api.complaints.download',$complaint->id) : '',
                    'is_resolved' => $complaint->is_resolved,
                    'created_at' => date('Y-m-d @ h:i:s A', strtotime($complaint->created_at)),
                    'updated_at' => date('Y-m-d @ h:i:s A', strtotime($complaint->updated_at)),
                ];
            }

            return response()->json(['complaints' => $result]);
        }

        return response()->json(['complaints' => []]);
    }
    //==================================================
    public function addComplaint(Request $request)
    {
        $user_id = $this->getUserByToken($request);
        //--- if the sent id doesn't match the id from db, 403 forbidden 
        if ($request->input('patient-id') != $user_id) {
            return response()->json(['error' => '403'], 403);
        }

        if ($request->getMethod() == 'POST') {

            $min_date = date('Y-m-d', strtotime("-1 week"));
            $max_date = date('Y-m-d');

            $rules = [
                'patient-id' => ['required', 'exists:patients,id'],
                'description' => ['required', 'max:5000', new GeneralText],
                'complaint-date' => ['required', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date],
                'attachment' => ['mimes:png,jpg,jpeg,mp4,m4v', 'min:10', 'max:20480'],
            ];

            $validator = Validator::make($request->all(), $rules);
            
            if ($validator->fails()) {

                //$errorString = implode("<br>", $validator->errors()->all());
            
                return response()->json(['error' => $validator->errors()->all()], 400);
            
            }

            $complaint = [
                'patient_id' => $request->input('patient-id'),
                'description' => $request->input('description'),
                'complaint_date' => date('Y-m-d', strtotime($request->input('complaint-date'))),
                'created_by' => 0
            ];

            if ($request->file('attachment') && $request->file('attachment')->isValid()) {

                $patient_dir = config('admin.complaints_dir') . '/' . $request->input('patient-id');

                $file_name = time() . '.' . $request->file('attachment')->guessExtension();

                $request->file('attachment')->storeAs($patient_dir, $file_name);

                $complaint['attachment'] = $file_name;
            }

            Complaint::create($complaint);

            return response()->json(['success' => 'success'],200);
        }
    }
    //==================================================
    public function downloadAttachment(Request $request, $complaint_id)
    {
        $complaint = Complaint::find($complaint_id);

        if ( $complaint && $complaint->id){

            $user_id = $this->getUserByToken($request);
            //--- if the sent id doesn't match the id from db, 403 forbidden 
            if ($complaint->patient_id != $user_id) {
                return response()->json(['error' => '403'], 403);
            }

            $path = config('admin.complaints_dir') . '/' . $complaint->patient_id . '/' . $complaint->attachment;

            if (Storage::exists($path)) {
    
                return Storage::download($path);
            }
        }

        return response()->json(['error' => '404'], 404);
    }
    //==================================================

    private function getUserByToken(Request $request)
    {
        //--- Get the token from request
        $header = $request->header('authorization');

        $token = str_replace('|', '', strstr($header, '|'));

        //--- Get the user id should be using this token
        $hashedToken = hash('sha256', $token);

        $db_token = PersonalAccessToken::where('token', $hashedToken)->first();

        $user = $db_token->tokenable;

        return $user->id;
    }
}
