<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class Complaint extends Model
{
    use HasFactory;

    protected $table = 'complaints';

    protected $fillable = [
        'patient_id','description', 'complaint_date', 'attachment', 'is_resolved','created_by'
    ];
    protected $casts = [
        'created_at'  => 'date:Y-m-d',
        'updated_at'  => 'date:Y-m-d',
        'complaint_date' => 'date:Y-m-d',
    ];

    //-------------------------------------------
    
    /**
     * Scope a query to filter by allowed patients only
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllowedPatients($query)
    {

        return (!is_admin() ) ? $query->whereIn('patient_id', allowed_patients()) : $query;
        
    }

    //-------------------------------------------
    public function patient(){

        return $this->belongsTo(Patient::class);
        
    }

    public function comments(){
        
        return $this->hasMany(ComplaintComments::class);
    
    }

    public function user(){
        return $this->belongsTo(User::class);
    }


    //-------------------------------------------
    // Attributes
    //-------------------------------------------
    public function getStatusAttribute(){

        return ($this->is_resolved) ? '<span class="text-success">' .__('admin/common.text.yes') .'</span>' : '<span class="text-danger">' .__('admin/common.text.no') .'</span>';
    
    }


    public function getCreatedTimeAttribute(){

        return date('Y-m-d @ h:i:s A', strtotime($this->created_at));

    }

    //--------------------------------------------
    public function getComplaintsSelect(){

        $patient = new Patient();

        $patient_name = $patient->patientFullName();

        $yes = __('admin/common.text.yes');
        $no = __('admin/common.text.no');

        $user_name = DB::raw("CONCAT(users.first_name,' ',users.last_name) AS user_name");
        
        $is_resolved = DB::raw("IF(is_resolved=0,'".$no."', '".$yes."') AS is_resolved");

        return ['complaints.id','complaint_date',$patient_name,'description','patient_id','attachment',$is_resolved,'complaints.created_at','complaints.updated_at', $user_name];

    }


}
