<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\HasApiTokens;

class Patient extends Model
{
    use HasFactory;use HasApiTokens;

    protected $table = 'patients';

    protected $fillable = [
        'first_name', 'second_name', 'third_name', 'last_name','dob', 
        'email', 'mobile', 'region_id', 'address', 'password','gender','device_token','is_verified','sms_count','created_by'
    ];

    protected $hidden = ['password'];

    /**
     * Scope a query to filter by allowed patients only
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllowedPatients($query)
    {

        return (!is_admin() ) ? $query->whereIn('patients.id', allowed_patients()) : $query;
        
    }

    //-----------------------------------------
    // Relations
    //-----------------------------------------
    public function appointments(){
        return $this->hasMany(Appointment::class);
    }

    public function complaints(){
        return $this->hasMany(Complaint::class);
    }

    public function region(){
        return $this->belongsTo(Region::class);
    }

    public function comments(){
        return $this->hasMany(PatientComment::class);
    }

    public function pretests(){
        //return $this->belongsToMany(PretestForm::class)->withPivot(['id','form','created_at','created_by']);
        return $this->hasMany(PatientPretest::class);
    }

    public function attachments(){
        return $this->hasMany(PatientAttachments::class);
    }

    public function reviews(){
        return $this->hasMany(Review::class);
    }


    //--------------------------------------------------
    // Attributes
    //--------------------------------------------------

    /**
     * Full name
     *
     * @return void
     */
    public function getFullName()
    {
        return "{$this->first_name} {$this->second_name} {$this->third_name} {$this->last_name}";
    }

    public function getFullNameAttribute()
    {
        return "{$this->first_name} {$this->second_name} {$this->third_name} {$this->last_name}";
    }
    /**
     * Patient Age
     *
     * @return void
     */
    public function getAgeAttribute()
    {
        return Carbon::parse($this->attributes['dob'])->age;
    }

    /**
     * Human readable gender text
     *
     * @return void
     */
    public function getFullGenderAttribute()
    {

        return (($this->gender == 'm') ? __('admin/common.gender.male') : __('admin/common.gender.female'));

    }
    public function getGender()
    {

        return (($this->gender == 'm') ? __('admin/common.gender.male') : __('admin/common.gender.female'));

    }

    public function getFormattedDobAttribute()
    {

        return date('Y-m-d', strtotime($this->attributes['dob'] ) );

    }

    //------------------------------------------------------------
    /**
     * Patient full name query ( for list view and select)
     *
     * @return void
     */
    public function patientFullName(){
        
        return DB::raw("CONCAT(COALESCE(patients.first_name,''),' ',COALESCE(patients.second_name,''),' ',COALESCE(patients.third_name,''),' ',COALESCE(patients.last_name,'')) AS patient_name");
    }

    public function getPatientInfo($id)
    {
        // Concatenating name columns
        
        $full_name = DB::raw("CONCAT(COALESCE(patients.first_name,''),' ',COALESCE(patients.second_name,''),' ',COALESCE(patients.third_name,''),' ',COALESCE(patients.last_name,'')) AS full_name");

        // Changing m and f to Male & Female
        $male = __('admin/common.gender.male');
        $female = __('admin/common.gender.female');

        $gender = DB::raw("IF(gender='m','" . $male . "', '" . $female . "') AS gender");

        $yes = __('admin/common.text.yes');
        $no = __('admin/common.text.no');

        $blocked = DB::raw("IF(is_blocked=true,'" . $yes . "', '" . $no . "') as is_blocked");

        // Calc age
        $age = DB::raw("DATE_FORMAT(NOW(), '%Y') - DATE_FORMAT(dob, '%Y') - (DATE_FORMAT(NOW(), '00-%m-%d') < DATE_FORMAT(dob, '00-%m-%d')) AS age");

        // Setting the parameters
        $select = ['patients.id', $full_name, $age,'dob','mobile', 'address', 'email', $gender, $blocked, 'regions.region as region'];

        $model = DB::table($this->table);

        $model = $model->select($select);
        
        $model = $model->join('regions', 'regions.id', '=', 'patients.region_id');

        $model = $model->where('patients.id', '=', $id);

        return $model->get()->first();
    }

    public function getPatientsDropDown(){
        // Concatinating name columns
        
        $full_name = DB::raw("CONCAT(COALESCE(patients.first_name,''),' ',COALESCE(patients.second_name,''),' ',COALESCE(patients.third_name,''),' ',COALESCE(patients.last_name,'')) AS full_name");
        
        $select = ['patients.id', $full_name];

        $model = DB::table($this->table);

        $model = $model->select($select);

        $model = $model->orderBy('full_name');

        return $model->get();
    }
}
